#!/bin/bash

echo "Querying Consul to set environment variables"

ENVCONSUL_OPTS="-sanitize -upcase"

if [ -n "$CONSUL_TOKEN" ]; then
  ENVCONSUL_OPTS="$ENVCONSUL_OPTS -consul-token $CONSUL_TOKEN"
fi

ENVCONSUL_OPTS="$ENVCONSUL_OPTS -consul-addr $CONSUL_SERVER"

# Watch the service-specific Consul prefix
if [ -z "$CONSUL_PATH" ]; then
  CONSUL_PATH="vamf/$NAMESPACE/$APP_NAME/$APP_VERSION"
fi

ENVCONSUL_OPTS="$ENVCONSUL_OPTS -prefix $CONSUL_PATH"

# Check for additional Consul paths and add as watch prefixes
if [ -n "$ADDITIONAL_CONSUL_PATHS" ]; then
  for val in ${ADDITIONAL_CONSUL_PATHS//,/ }
  do
    ENVCONSUL_OPTS="$ENVCONSUL_OPTS -prefix $val"
  done
fi

# If a Vault address is specified then we should attempt to use Vault for secrets
if [ -n "${VAULT_ADDR}" ]; then
  echo "Using Vault for app secrets"

  until curl --head $VAULT_ADDR/v1/sys/health; do
    echo waiting for Vault at $VAULT_ADDR
    sleep 3
  done

  # Get a Vault token for the AppRole specified.
  export VAULT_TOKEN=$(curl -X POST -d '{"role_id":"'$NAMESPACE'-read" }' $VAULT_ADDR/v1/auth/approle/login | jq -r '.auth.client_token')

  # The Vault address has to be set in a config file, cannot be passed as a CLI paramater
  echo vault '{ address = "'$VAULT_ADDR'" token = "'$VAULT_TOKEN'" renew_token = true }' > /tmp/vault_config.hcl

  # Some services don't have secrets
  if [ -n "$SECRET_PATH" ]; then
    echo secret '{ path = "'$SECRET_PATH'" no_prefix = true }' >> /tmp/vault_config.hcl
  fi

  # Add the Vault configuration to the envconsul options
  ENVCONSUL_OPTS="$ENVCONSUL_OPTS -config /tmp/vault_config.hcl"
else
  echo "Not using Vault for app secrets"
fi

# Splay Option
if [ -n "$SPLAY_DURATION" ]; then
  ENVCONSUL_OPTS="$ENVCONSUL_OPTS -exec-splay=$SPLAY_DURATION"
else
  ENVCONSUL_OPTS="$ENVCONSUL_OPTS -exec-splay=1s"
fi

# Start the Java process via envconsul using -once option to prevent polling
exec envconsul $ENVCONSUL_OPTS -once /opt/vamf/service/run-service.sh
